<?php
/**
 * API Entry Point
 * Main bootstrap file for the RESTful API
 */

// Error handling
ini_set('display_errors', 0);
error_reporting(E_ALL);

set_exception_handler(function ($exception) {
    Logger::critical('Uncaught exception', [
        'message' => $exception->getMessage(),
        'file' => $exception->getFile(),
        'line' => $exception->getLine(),
        'trace' => $exception->getTraceAsString()
    ]);
    
    if (defined('APP_DEBUG') && APP_DEBUG) {
        Response::serverError($exception->getMessage());
    } else {
        Response::serverError('Internal server error');
    }
});

set_error_handler(function ($errno, $errstr, $errfile, $errline) {
    throw new ErrorException($errstr, 0, $errno, $errfile, $errline);
});

// Load configuration
require_once __DIR__ . '/../config/env.php';

// Load autoloader
require_once __DIR__ . '/../config/autoload.php';

// Security headers
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: DENY');
header('X-XSS-Protection: 1; mode=block');
header('Referrer-Policy: strict-origin-when-cross-origin');

// Initialize request
$request = new Request();

// Initialize router
$router = new Router();

// Register global middleware
$router->middleware('cors', [CorsMiddleware::class, 'handle']);
$router->middleware('auth', [AuthMiddleware::class, 'handle']);
$router->middleware('rateLimit', [RateLimitMiddleware::class, 'handle']);

// Apply CORS to all routes
CorsMiddleware::handle($request);

// ============================================================================
// ROUTES DEFINITION
// ============================================================================

// Health check
$router->get('/health', function ($request) {
    Response::success([
        'status' => 'healthy',
        'timestamp' => date('c'),
        'version' => API_VERSION
    ]);
});

// API version group
$router->group(['prefix' => '/v1'], function ($router) {
    
    // ========================================================================
    // PUBLIC ROUTES (No authentication required)
    // ========================================================================
    
    // Authentication
    $router->post('/auth/login', 'AuthController@login');
    $router->post('/auth/register', 'AuthController@register');
    $router->post('/auth/forgot-password', 'AuthController@forgotPassword');
    $router->post('/auth/reset-password', 'AuthController@resetPassword');
    
    // Public services
    $router->get('/services', 'ServiceController@index');
    $router->get('/services/{id}', 'ServiceController@show');
    $router->get('/packages', 'PackageController@index');
    $router->get('/packages/{id}', 'PackageController@show');
    
    // Webhooks (public but verified internally)
    $router->post('/webhooks/xendit', 'WebhookController@xendit');
    $router->post('/webhooks/tripay', 'WebhookController@tripay');
    
    // ========================================================================
    // PROTECTED ROUTES (Authentication required)
    // ========================================================================
    
    $router->group(['middleware' => ['auth']], function ($router) {
        
        // Auth
        $router->post('/auth/refresh', 'AuthController@refresh');
        $router->post('/auth/logout', 'AuthController@logout');
        $router->get('/auth/me', 'AuthController@me');
        
        // Customer profile
        $router->get('/profile', 'ProfileController@show');
        $router->put('/profile', 'ProfileController@update');
        $router->post('/profile/change-password', 'ProfileController@changePassword');
        
        // Customer invoices
        $router->get('/invoices', 'InvoiceController@index');
        $router->get('/invoices/{id}', 'InvoiceController@show');
        $router->get('/invoices/{id}/download', 'InvoiceController@download');
        
        // Payments
        $router->get('/payment-channels', 'PaymentController@channels');
        $router->get('/payment-gateways', 'PaymentController@gateways');
        $router->post('/payments/create', 'PaymentController@create');
        $router->get('/payments/{id}', 'PaymentController@show');
        $router->get('/payments/{id}/status', 'PaymentController@status');
        
        // Notifications
        $router->post('/notifications/register-token', 'NotificationController@registerToken');
        $router->post('/notifications/unregister-token', 'NotificationController@unregisterToken');
        $router->get('/notifications', 'NotificationController@index');
        $router->put('/notifications/{id}/read', 'NotificationController@markAsRead');
        
        // Tickets
        $router->get('/tickets', 'TicketController@index');
        $router->post('/tickets', 'TicketController@create');
        $router->get('/tickets/{id}', 'TicketController@show');
        $router->post('/tickets/{id}/reply', 'TicketController@reply');
        
        // Collector routes (role-specific)
        $router->group(['prefix' => '/collector'], function ($router) {
            $router->get('/customers', 'CollectorController@customers');
            $router->get('/customers/due', 'CollectorController@dueCustomers');
            $router->get('/customers/map', 'CollectorController@mapData');
            $router->post('/payments/cash', 'CollectorController@recordCashPayment');
            $router->get('/performance', 'CollectorController@performance');
        });
        
        // Admin routes (admin only)
        $router->group(['prefix' => '/admin'], function ($router) {
            $router->get('/dashboard', 'AdminController@dashboard');
            
            // Users
            $router->get('/users', 'UserController@index');
            $router->post('/users', 'UserController@create');
            $router->get('/users/{id}', 'UserController@show');
            $router->put('/users/{id}', 'UserController@update');
            $router->delete('/users/{id}', 'UserController@delete');
            
            // Customers
            $router->get('/customers', 'CustomerController@index');
            $router->post('/customers', 'CustomerController@create');
            $router->get('/customers/{id}', 'CustomerController@show');
            $router->put('/customers/{id}', 'CustomerController@update');
            $router->delete('/customers/{id}', 'CustomerController@delete');
            
            // Services & Packages
            $router->post('/services', 'ServiceController@create');
            $router->put('/services/{id}', 'ServiceController@update');
            $router->delete('/services/{id}', 'ServiceController@delete');
            
            // Billing
            $router->post('/invoices/generate', 'InvoiceController@generate');
            $router->put('/invoices/{id}/cancel', 'InvoiceController@cancel');
            
            // Reports
            $router->get('/reports/invoices', 'ReportController@invoices');
            $router->get('/reports/payments', 'ReportController@payments');
            $router->get('/reports/customers', 'ReportController@customers');
        });
    });
});

// ============================================================================
// DISPATCH REQUEST
// ============================================================================

try {
    $router->dispatch($request);
} catch (Exception $e) {
    Logger::critical('Router dispatch failed', [
        'error' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
    ]);
    
    Response::serverError('Internal server error');
}
