<?php
/**
 * Authentication Controller
 * Handles authentication endpoints
 */
class AuthController
{
    private $authService;
    private $validator;
    private $request;
    private $logger;

    public function __construct()
    {
        $this->authService = new AuthService();
        $this->validator = new Validator();
        $this->request = new Request();
        $this->logger = Logger::getInstance();
    }

    /**
     * Login endpoint
     * POST /api/auth/login
     */
    public function login()
    {
        try {
            $data = $this->request->getBody();

            // Validate input
            $rules = [
                'email' => 'required|email',
                'password' => 'required|min:6'
            ];

            $errors = $this->validator->validate($data, $rules);
            
            if (!empty($errors)) {
                return Response::validationError($errors);
            }

            // Authenticate user
            $result = $this->authService->login(
                $data['email'],
                $data['password'],
                $this->request->getIpAddress(),
                $this->request->getUserAgent()
            );

            return Response::success($result, 'Login successful');

        } catch (AuthenticationException $e) {
            return Response::unauthorized($e->getMessage());
        } catch (ValidationException $e) {
            return Response::validationError($e->getDetails(), $e->getMessage());
        } catch (Exception $e) {
            $this->logger->error('Login endpoint error', ['error' => $e->getMessage()]);
            return Response::serverError('Authentication failed');
        }
    }

    /**
     * Register endpoint
     * POST /api/auth/register
     */
    public function register()
    {
        try {
            $data = $this->request->getBody();

            // Validate input
            $rules = [
                'email' => 'required|email',
                'password' => 'required|min:8',
                'full_name' => 'required|min:3',
                'phone' => 'phone'
            ];

            $errors = $this->validator->validate($data, $rules);
            
            if (!empty($errors)) {
                return Response::validationError($errors);
            }

            // Check password strength (optional additional check)
            if (!$this->isStrongPassword($data['password'])) {
                return Response::validationError([
                    'password' => 'Password must contain at least one uppercase, one lowercase, and one number'
                ]);
            }

            // Register user
            $user = $this->authService->register($data);

            // Auto-login after registration
            $loginResult = $this->authService->login(
                $data['email'],
                $data['password'],
                $this->request->getIpAddress(),
                $this->request->getUserAgent()
            );

            return Response::created($loginResult, 'Registration successful');

        } catch (ValidationException $e) {
            return Response::validationError($e->getDetails(), $e->getMessage());
        } catch (DatabaseException $e) {
            return Response::serverError('Registration failed. Please try again.');
        } catch (Exception $e) {
            $this->logger->error('Register endpoint error', ['error' => $e->getMessage()]);
            return Response::serverError('Registration failed');
        }
    }

    /**
     * Refresh token endpoint
     * POST /api/auth/refresh
     */
    public function refresh()
    {
        try {
            $data = $this->request->getBody();

            if (empty($data['refresh_token'])) {
                return Response::badRequest('Refresh token is required');
            }

            // Refresh tokens
            $result = $this->authService->refreshToken($data['refresh_token']);

            return Response::success($result, 'Token refreshed successfully');

        } catch (AuthenticationException $e) {
            return Response::unauthorized($e->getMessage());
        } catch (Exception $e) {
            $this->logger->error('Refresh endpoint error', ['error' => $e->getMessage()]);
            return Response::serverError('Token refresh failed');
        }
    }

    /**
     * Logout endpoint
     * POST /api/auth/logout
     */
    public function logout()
    {
        try {
            $data = $this->request->getBody();
            $userId = $this->request->getAttribute('user_id');

            if (!$userId) {
                return Response::unauthorized('Not authenticated');
            }

            $refreshToken = $data['refresh_token'] ?? null;

            // Logout user
            $this->authService->logout($userId, $refreshToken);

            return Response::success(null, 'Logout successful');

        } catch (Exception $e) {
            $this->logger->error('Logout endpoint error', ['error' => $e->getMessage()]);
            return Response::serverError('Logout failed');
        }
    }

    /**
     * Get current user endpoint
     * GET /api/auth/me
     */
    public function me()
    {
        try {
            $userId = $this->request->getAttribute('user_id');

            if (!$userId) {
                return Response::unauthorized('Not authenticated');
            }

            $userRepo = new UserRepository();
            $user = $userRepo->findByIdWithPermissions($userId);

            if (!$user) {
                return Response::notFound('User not found');
            }

            return Response::success($user->toArray(), 'User retrieved successfully');

        } catch (Exception $e) {
            $this->logger->error('Me endpoint error', ['error' => $e->getMessage()]);
            return Response::serverError('Failed to retrieve user');
        }
    }

    /**
     * Change password endpoint
     * POST /api/auth/change-password
     */
    public function changePassword()
    {
        try {
            $data = $this->request->getBody();
            $userId = $this->request->getAttribute('user_id');

            if (!$userId) {
                return Response::unauthorized('Not authenticated');
            }

            // Validate input
            $rules = [
                'old_password' => 'required',
                'new_password' => 'required|min:8',
                'confirm_password' => 'required'
            ];

            $errors = $this->validator->validate($data, $rules);
            
            if (!empty($errors)) {
                return Response::validationError($errors);
            }

            // Check password confirmation
            if ($data['new_password'] !== $data['confirm_password']) {
                return Response::validationError([
                    'confirm_password' => 'Passwords do not match'
                ]);
            }

            // Change password
            $this->authService->changePassword(
                $userId,
                $data['old_password'],
                $data['new_password']
            );

            return Response::success(null, 'Password changed successfully');

        } catch (ValidationException $e) {
            return Response::validationError(['old_password' => $e->getMessage()]);
        } catch (Exception $e) {
            $this->logger->error('Change password endpoint error', ['error' => $e->getMessage()]);
            return Response::serverError('Failed to change password');
        }
    }

    /**
     * Check password strength
     */
    private function isStrongPassword($password)
    {
        // At least one uppercase, one lowercase, one number
        return preg_match('/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d).+$/', $password);
    }
}
