<?php
/**
 * Collector Controller
 * Handles collector-specific endpoints for field staff
 */
class CollectorController
{
    private $request;
    private $validator;
    private $logger;
    private $invoiceRepo;
    private $paymentRepo;
    private $customerRepo;

    public function __construct()
    {
        $this->request = new Request();
        $this->validator = new Validator();
        $this->logger = Logger::getInstance();
        $this->invoiceRepo = new InvoiceRepository();
        $this->paymentRepo = new PaymentRepository();
        $this->customerRepo = new CustomerRepository();
    }

    /**
     * Get customers assigned to collector
     * GET /api/collector/customers
     */
    public function customers()
    {
        try {
            $user = $this->request->getUser();
            $page = $this->request->getQueryParam('page', 1);
            $perPage = $this->request->getQueryParam('per_page', 20);
            $search = $this->request->getQueryParam('search', '');
            
            $db = Database::getInstance();
            $offset = ($page - 1) * $perPage;
            
            // Get customers with pending invoices
            $query = "SELECT c.*, 
                      COUNT(DISTINCT i.id) as pending_invoices_count,
                      SUM(CASE WHEN i.status IN ('pending', 'overdue') THEN i.total_amount ELSE 0 END) as total_due
                      FROM customers c
                      LEFT JOIN invoices i ON c.id = i.customer_id AND i.status IN ('pending', 'overdue')
                      WHERE c.status = 'active'";
            
            if ($search) {
                $query .= " AND (c.full_name LIKE ? OR c.customer_code LIKE ? OR c.phone LIKE ?)";
            }
            
            $query .= " GROUP BY c.id ORDER BY pending_invoices_count DESC, c.full_name ASC LIMIT ? OFFSET ?";
            
            $params = $search ? 
                ["%$search%", "%$search%", "%$search%", $perPage, $offset] : 
                [$perPage, $offset];
            
            $customers = $db->query($query, $params);
            
            // Get total count
            $countQuery = "SELECT COUNT(DISTINCT c.id) as total FROM customers c WHERE c.status = 'active'";
            if ($search) {
                $countQuery .= " AND (c.full_name LIKE ? OR c.customer_code LIKE ? OR c.phone LIKE ?)";
            }
            $countParams = $search ? ["%$search%", "%$search%", "%$search%"] : [];
            $totalResult = $db->query($countQuery, $countParams);
            $total = $totalResult[0]['total'] ?? 0;
            
            return Response::success([
                'customers' => $customers,
                'pagination' => [
                    'current_page' => (int)$page,
                    'per_page' => (int)$perPage,
                    'total' => (int)$total,
                    'total_pages' => ceil($total / $perPage)
                ]
            ]);

        } catch (Exception $e) {
            $this->logger->error('Get collector customers error', ['error' => $e->getMessage()]);
            return Response::serverError('Failed to retrieve customers');
        }
    }

    /**
     * Get customers with due/overdue invoices
     * GET /api/collector/customers/due
     */
    public function dueCustomers()
    {
        try {
            $user = $this->request->getUser();
            $page = $this->request->getQueryParam('page', 1);
            $perPage = $this->request->getQueryParam('per_page', 20);
            
            $db = Database::getInstance();
            $offset = ($page - 1) * $perPage;
            
            $query = "SELECT c.*, 
                      i.id as invoice_id,
                      i.invoice_number,
                      i.billing_period_start,
                      i.billing_period_end,
                      i.due_date,
                      i.total_amount,
                      i.status as invoice_status,
                      DATEDIFF(CURDATE(), i.due_date) as days_overdue
                      FROM customers c
                      INNER JOIN invoices i ON c.id = i.customer_id
                      WHERE i.status IN ('pending', 'overdue')
                      AND c.status = 'active'
                      ORDER BY days_overdue DESC, i.due_date ASC
                      LIMIT ? OFFSET ?";
            
            $customers = $db->query($query, [$perPage, $offset]);
            
            // Get total count
            $countQuery = "SELECT COUNT(DISTINCT i.id) as total 
                          FROM customers c
                          INNER JOIN invoices i ON c.id = i.customer_id
                          WHERE i.status IN ('pending', 'overdue') AND c.status = 'active'";
            $totalResult = $db->query($countQuery);
            $total = $totalResult[0]['total'] ?? 0;
            
            return Response::success([
                'customers' => $customers,
                'pagination' => [
                    'current_page' => (int)$page,
                    'per_page' => (int)$perPage,
                    'total' => (int)$total,
                    'total_pages' => ceil($total / $perPage)
                ]
            ]);

        } catch (Exception $e) {
            $this->logger->error('Get due customers error', ['error' => $e->getMessage()]);
            return Response::serverError('Failed to retrieve due customers');
        }
    }

    /**
     * Get map data for customers
     * GET /api/collector/customers/map
     */
    public function mapData()
    {
        try {
            $user = $this->request->getUser();
            
            $db = Database::getInstance();
            
            $query = "SELECT c.id, c.customer_code, c.full_name, c.address,
                      c.latitude, c.longitude, c.phone,
                      COUNT(DISTINCT i.id) as pending_count,
                      SUM(CASE WHEN i.status IN ('pending', 'overdue') THEN i.total_amount ELSE 0 END) as total_due
                      FROM customers c
                      LEFT JOIN invoices i ON c.id = i.customer_id AND i.status IN ('pending', 'overdue')
                      WHERE c.status = 'active' 
                      AND c.latitude IS NOT NULL 
                      AND c.longitude IS NOT NULL
                      GROUP BY c.id";
            
            $customers = $db->query($query);
            
            // Convert to GeoJSON format
            $features = [];
            foreach ($customers as $customer) {
                $features[] = [
                    'type' => 'Feature',
                    'geometry' => [
                        'type' => 'Point',
                        'coordinates' => [
                            (float)$customer['longitude'],
                            (float)$customer['latitude']
                        ]
                    ],
                    'properties' => [
                        'id' => $customer['id'],
                        'customer_code' => $customer['customer_code'],
                        'full_name' => $customer['full_name'],
                        'address' => $customer['address'],
                        'phone' => $customer['phone'],
                        'pending_count' => (int)$customer['pending_count'],
                        'total_due' => (float)$customer['total_due']
                    ]
                ];
            }
            
            $geoJson = [
                'type' => 'FeatureCollection',
                'features' => $features
            ];
            
            return Response::success($geoJson);

        } catch (Exception $e) {
            $this->logger->error('Get map data error', ['error' => $e->getMessage()]);
            return Response::serverError('Failed to retrieve map data');
        }
    }

    /**
     * Record cash payment
     * POST /api/collector/payments/cash
     */
    public function recordCashPayment()
    {
        try {
            $user = $this->request->getUser();
            $data = $this->request->getBody();

            // Validate input
            $rules = [
                'invoice_id' => 'required|uuid',
                'amount' => 'required|numeric|min:0.01',
                'latitude' => 'numeric',
                'longitude' => 'numeric',
                'payment_proof_url' => 'url'
            ];

            $errors = $this->validator->validate($data, $rules);
            
            if (!empty($errors)) {
                return Response::validationError($errors);
            }

            // Get invoice
            $invoice = $this->invoiceRepo->findById($data['invoice_id']);
            
            if (!$invoice) {
                return Response::notFound('Invoice not found');
            }

            if ($invoice->isPaid()) {
                return Response::badRequest('Invoice is already paid');
            }

            // Start transaction
            $db = Database::getInstance();
            $db->beginTransaction();

            try {
                // Get cash gateway
                $gatewayQuery = "SELECT id FROM payment_gateways WHERE code = 'CASH' LIMIT 1";
                $gatewayResult = $db->query($gatewayQuery);
                
                if (empty($gatewayResult)) {
                    throw new Exception('Cash payment gateway not configured');
                }
                
                $gatewayId = $gatewayResult[0]['id'];

                // Create payment record
                $paymentNumber = 'PAY-' . date('Ymd') . '-' . strtoupper(substr(uniqid(), -6));
                
                $paymentData = [
                    'payment_number' => $paymentNumber,
                    'invoice_id' => $data['invoice_id'],
                    'customer_id' => $invoice->getCustomerId(),
                    'gateway_id' => $gatewayId,
                    'payment_type' => 'cash',
                    'amount' => $data['amount'],
                    'admin_fee' => 0,
                    'total_amount' => $data['amount'],
                    'status' => 'paid',
                    'collector_id' => $user['id'],
                    'paid_at' => date('Y-m-d H:i:s'),
                    'payment_proof_url' => $data['payment_proof_url'] ?? null,
                    'notes' => $data['notes'] ?? null
                ];

                $payment = $this->paymentRepo->create($paymentData);

                // Record collector payment with GPS
                $collectorPaymentQuery = "INSERT INTO collector_payments 
                    (collector_id, payment_id, latitude, longitude, notes, collected_at) 
                    VALUES (?, ?, ?, ?, ?, NOW())";
                
                $db->execute($collectorPaymentQuery, [
                    $user['id'],
                    $payment->getId(),
                    $data['latitude'] ?? null,
                    $data['longitude'] ?? null,
                    $data['notes'] ?? null
                ]);

                // Mark invoice as paid
                $billingService = new BillingService();
                $billingService->markAsPaid($data['invoice_id']);

                $db->commit();

                // Send notification to customer
                $notificationService = new NotificationService();
                $notificationService->sendPaymentConfirmation(
                    $invoice->getCustomerId(),
                    $payment->getId(),
                    $data['amount']
                );

                $this->logger->info('Cash payment recorded by collector', [
                    'payment_id' => $payment->getId(),
                    'invoice_id' => $data['invoice_id'],
                    'collector_id' => $user['id']
                ]);

                return Response::created($payment->toArray(), 'Payment recorded successfully');

            } catch (Exception $e) {
                $db->rollback();
                throw $e;
            }

        } catch (NotFoundException $e) {
            return Response::notFound($e->getMessage());
        } catch (Exception $e) {
            $this->logger->error('Record cash payment error', ['error' => $e->getMessage()]);
            return Response::serverError('Failed to record payment');
        }
    }

    /**
     * Get collector performance
     * GET /api/collector/performance
     */
    public function performance()
    {
        try {
            $user = $this->request->getUser();
            $period = $this->request->getQueryParam('period', 'month'); // day, week, month, year
            
            $db = Database::getInstance();
            
            // Calculate date range
            $dateCondition = '';
            switch ($period) {
                case 'day':
                    $dateCondition = 'DATE(p.paid_at) = CURDATE()';
                    break;
                case 'week':
                    $dateCondition = 'YEARWEEK(p.paid_at) = YEARWEEK(CURDATE())';
                    break;
                case 'year':
                    $dateCondition = 'YEAR(p.paid_at) = YEAR(CURDATE())';
                    break;
                case 'month':
                default:
                    $dateCondition = 'YEAR(p.paid_at) = YEAR(CURDATE()) AND MONTH(p.paid_at) = MONTH(CURDATE())';
            }
            
            $query = "SELECT 
                      COUNT(DISTINCT p.id) as total_collections,
                      SUM(p.total_amount) as total_amount,
                      COUNT(DISTINCT p.customer_id) as unique_customers,
                      AVG(p.total_amount) as average_amount
                      FROM payments p
                      WHERE p.collector_id = ? 
                      AND p.status = 'paid'
                      AND $dateCondition";
            
            $result = $db->query($query, [$user['id']]);
            $stats = $result[0] ?? [
                'total_collections' => 0,
                'total_amount' => 0,
                'unique_customers' => 0,
                'average_amount' => 0
            ];
            
            // Get recent collections
            $recentQuery = "SELECT p.*, i.invoice_number, c.full_name as customer_name
                           FROM payments p
                           INNER JOIN invoices i ON p.invoice_id = i.id
                           INNER JOIN customers c ON p.customer_id = c.id
                           WHERE p.collector_id = ? AND p.status = 'paid'
                           ORDER BY p.paid_at DESC
                           LIMIT 10";
            
            $recentCollections = $db->query($recentQuery, [$user['id']]);
            
            return Response::success([
                'stats' => $stats,
                'recent_collections' => $recentCollections,
                'period' => $period
            ]);

        } catch (Exception $e) {
            $this->logger->error('Get collector performance error', ['error' => $e->getMessage()]);
            return Response::serverError('Failed to retrieve performance data');
        }
    }
}
