<?php
/**
 * Payment Controller
 * Handles payment endpoints
 */
class PaymentController
{
    private $paymentGatewayService;
    private $validator;
    private $request;
    private $logger;

    public function __construct()
    {
        $this->paymentGatewayService = new PaymentGatewayService();
        $this->validator = new Validator();
        $this->request = new Request();
        $this->logger = Logger::getInstance();
    }

    /**
     * Create payment
     * POST /api/payments
     */
    public function create()
    {
        try {
            $data = $this->request->getBody();

            // Validate input
            $rules = [
                'invoice_id' => 'required|uuid',
                'gateway_id' => 'required|uuid',
                'channel_id' => 'uuid'
            ];

            $errors = $this->validator->validate($data, $rules);
            
            if (!empty($errors)) {
                return Response::validationError($errors);
            }

            // Create payment
            $payment = $this->paymentGatewayService->createPayment(
                $data['invoice_id'],
                $data['gateway_id'],
                $data['channel_id'] ?? null
            );

            return Response::created($payment->toArray(), 'Payment created successfully');

        } catch (NotFoundException $e) {
            return Response::notFound($e->getMessage());
        } catch (PaymentException $e) {
            return Response::badRequest($e->getMessage());
        } catch (Exception $e) {
            $this->logger->error('Create payment error', ['error' => $e->getMessage()]);
            return Response::serverError('Failed to create payment');
        }
    }

    /**
     * Get payment by ID
     * GET /api/payments/{id}
     */
    public function show($id)
    {
        try {
            $payment = $this->paymentGatewayService->getPayment($id);

            return Response::success($payment->toArray(), 'Payment retrieved successfully');

        } catch (NotFoundException $e) {
            return Response::notFound($e->getMessage());
        } catch (Exception $e) {
            $this->logger->error('Get payment error', ['error' => $e->getMessage()]);
            return Response::serverError('Failed to retrieve payment');
        }
    }

    /**
     * Get payment channels
     * GET /api/payments/channels
     */
    public function channels()
    {
        try {
            $gatewayId = $this->request->getQueryParam('gateway_id');
            
            $channels = $this->paymentGatewayService->getPaymentChannels($gatewayId);

            return Response::success($channels, 'Payment channels retrieved successfully');

        } catch (Exception $e) {
            $this->logger->error('Get payment channels error', ['error' => $e->getMessage()]);
            return Response::serverError('Failed to retrieve payment channels');
        }
    }

    /**
     * Get active gateways
     * GET /api/payments/gateways
     */
    public function gateways()
    {
        try {
            $gateways = $this->paymentGatewayService->getActiveGateways();

            return Response::success($gateways, 'Payment gateways retrieved successfully');

        } catch (Exception $e) {
            $this->logger->error('Get payment gateways error', ['error' => $e->getMessage()]);
            return Response::serverError('Failed to retrieve payment gateways');
        }
    }
}
