<?php
/**
 * Logger Class
 * PSR-3 compliant logging system
 */

class Logger {
    private const EMERGENCY = 'emergency';
    private const ALERT     = 'alert';
    private const CRITICAL  = 'critical';
    private const ERROR     = 'error';
    private const WARNING   = 'warning';
    private const NOTICE    = 'notice';
    private const INFO      = 'info';
    private const DEBUG     = 'debug';
    
    private static $levels = [
        self::DEBUG     => 0,
        self::INFO      => 1,
        self::NOTICE    => 2,
        self::WARNING   => 3,
        self::ERROR     => 4,
        self::CRITICAL  => 5,
        self::ALERT     => 6,
        self::EMERGENCY => 7,
    ];
    
    /**
     * Log a message
     */
    private static function log(string $level, string $message, array $context = []): void {
        $minLevel = defined('LOG_LEVEL') ? LOG_LEVEL : 'debug';
        
        if (self::$levels[$level] < self::$levels[$minLevel]) {
            return;
        }
        
        $logPath = defined('LOG_PATH') ? LOG_PATH : __DIR__ . '/../logs/';
        
        if (!is_dir($logPath)) {
            mkdir($logPath, 0755, true);
        }
        
        $filename = $logPath . date('Y-m-d') . '.log';
        $timestamp = date('Y-m-d H:i:s');
        $contextStr = !empty($context) ? json_encode($context, JSON_UNESCAPED_SLASHES) : '';
        
        $logEntry = sprintf(
            "[%s] %s: %s %s\n",
            $timestamp,
            strtoupper($level),
            $message,
            $contextStr
        );
        
        file_put_contents($filename, $logEntry, FILE_APPEND);
        
        // Also log to database for critical errors
        if (self::$levels[$level] >= self::$levels[self::ERROR]) {
            self::logToDatabase($level, $message, $context);
        }
    }
    
    private static $isLoggingToDatabase = false;
    
    /**
     * Log to database (with recursion protection)
     */
    private static function logToDatabase(string $level, string $message, array $context): void {
        // Prevent infinite recursion
        if (self::$isLoggingToDatabase) {
            return;
        }
        
        self::$isLoggingToDatabase = true;
        
        try {
            $db = Database::getInstance();
            $db->insert('system_logs', [
                'id' => $db->uuid(),
                'level' => $level,
                'message' => $message,
                'context' => json_encode($context),
                'source' => $context['source'] ?? 'system'
            ]);
        } catch (Exception $e) {
            // Fail silently to prevent recursion
        } finally {
            self::$isLoggingToDatabase = false;
        }
    }
    
    public static function emergency(string $message, array $context = []): void {
        self::log(self::EMERGENCY, $message, $context);
    }
    
    public static function alert(string $message, array $context = []): void {
        self::log(self::ALERT, $message, $context);
    }
    
    public static function critical(string $message, array $context = []): void {
        self::log(self::CRITICAL, $message, $context);
    }
    
    public static function error(string $message, array $context = []): void {
        self::log(self::ERROR, $message, $context);
    }
    
    public static function warning(string $message, array $context = []): void {
        self::log(self::WARNING, $message, $context);
    }
    
    public static function notice(string $message, array $context = []): void {
        self::log(self::NOTICE, $message, $context);
    }
    
    public static function info(string $message, array $context = []): void {
        self::log(self::INFO, $message, $context);
    }
    
    public static function debug(string $message, array $context = []): void {
        self::log(self::DEBUG, $message, $context);
    }
}
