<?php
/**
 * Request Class
 * HTTP request handler with input sanitization
 */

class Request {
    private $method;
    private $uri;
    private $headers;
    private $body;
    private $queryParams;
    private $routeParams;
    private $files;
    private $ip;
    private $userAgent;
    
    public function __construct() {
        $this->method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
        $this->uri = parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH);
        $this->headers = $this->getRequestHeaders();
        $this->body = $this->getRequestBody();
        $this->queryParams = $_GET ?? [];
        $this->routeParams = [];
        $this->files = $_FILES ?? [];
        $this->ip = $this->getClientIp();
        $this->userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
    }
    
    /**
     * Get HTTP method
     */
    public function getMethod(): string {
        return $this->method;
    }
    
    /**
     * Get request URI
     */
    public function getUri(): string {
        return $this->uri;
    }
    
    /**
     * Get request headers
     */
    private function getRequestHeaders(): array {
        $headers = [];
        foreach ($_SERVER as $key => $value) {
            if (strpos($key, 'HTTP_') === 0) {
                $header = str_replace(' ', '-', ucwords(str_replace('_', ' ', strtolower(substr($key, 5)))));
                $headers[$header] = $value;
            }
        }
        return $headers;
    }
    
    /**
     * Get specific header
     */
    public function getHeader(string $name): ?string {
        return $this->headers[$name] ?? null;
    }
    
    /**
     * Get all headers
     */
    public function getAllHeaders(): array {
        return $this->headers;
    }
    
    /**
     * Get request body
     */
    private function getRequestBody(): array {
        if ($this->method === 'GET') {
            return [];
        }
        
        $contentType = $_SERVER['CONTENT_TYPE'] ?? '';
        
        if (strpos($contentType, 'application/json') !== false) {
            $input = file_get_contents('php://input');
            $decoded = json_decode($input, true);
            return is_array($decoded) ? $decoded : [];
        }
        
        return $_POST ?? [];
    }
    
    /**
     * Get input value
     */
    public function input(string $key, $default = null) {
        return $this->body[$key] ?? $default;
    }
    
    /**
     * Get all input
     */
    public function all(): array {
        return $this->body;
    }
    
    /**
     * Check if input exists
     */
    public function has(string $key): bool {
        return isset($this->body[$key]);
    }
    
    /**
     * Get query parameter
     */
    public function query(string $key, $default = null) {
        return $this->queryParams[$key] ?? $default;
    }
    
    /**
     * Get all query parameters
     */
    public function allQuery(): array {
        return $this->queryParams;
    }
    
    /**
     * Set route parameters
     */
    public function setRouteParams(array $params): void {
        $this->routeParams = $params;
    }
    
    /**
     * Get route parameter
     */
    public function param(string $key, $default = null) {
        return $this->routeParams[$key] ?? $default;
    }
    
    /**
     * Get uploaded file
     */
    public function file(string $key): ?array {
        return $this->files[$key] ?? null;
    }
    
    /**
     * Get client IP address
     */
    private function getClientIp(): string {
        $headers = [
            'HTTP_CLIENT_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_FORWARDED',
            'HTTP_X_CLUSTER_CLIENT_IP',
            'HTTP_FORWARDED_FOR',
            'HTTP_FORWARDED',
            'REMOTE_ADDR'
        ];
        
        foreach ($headers as $header) {
            if (!empty($_SERVER[$header])) {
                $ips = explode(',', $_SERVER[$header]);
                return trim($ips[0]);
            }
        }
        
        return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }
    
    /**
     * Get client IP
     */
    public function getIp(): string {
        return $this->ip;
    }
    
    /**
     * Get user agent
     */
    public function getUserAgent(): string {
        return $this->userAgent;
    }
    
    /**
     * Get bearer token from Authorization header
     */
    public function bearerToken(): ?string {
        $auth = $this->getHeader('Authorization');
        
        if ($auth && preg_match('/Bearer\s+(.*)$/i', $auth, $matches)) {
            return $matches[1];
        }
        
        return null;
    }
    
    /**
     * Check if request expects JSON
     */
    public function expectsJson(): bool {
        $accept = $this->getHeader('Accept') ?? '';
        return strpos($accept, 'application/json') !== false;
    }
    
    /**
     * Check if request is AJAX
     */
    public function isAjax(): bool {
        return $this->getHeader('X-Requested-With') === 'XMLHttpRequest';
    }
    
    /**
     * Sanitize input
     */
    public function sanitize(string $key, string $type = 'string') {
        $value = $this->input($key);
        
        if ($value === null) {
            return null;
        }
        
        switch ($type) {
            case 'int':
                return (int) $value;
            case 'float':
                return (float) $value;
            case 'bool':
                return (bool) $value;
            case 'email':
                return filter_var($value, FILTER_SANITIZE_EMAIL);
            case 'url':
                return filter_var($value, FILTER_SANITIZE_URL);
            case 'string':
            default:
                return htmlspecialchars(strip_tags($value), ENT_QUOTES, 'UTF-8');
        }
    }
}
