<?php
/**
 * Response Class
 * Standardized JSON API responses
 */

class Response {
    /**
     * Send JSON response
     */
    public static function json(array $data, int $statusCode = 200): void {
        http_response_code($statusCode);
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode($data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        exit;
    }
    
    /**
     * Success response
     */
    public static function success($data = null, string $message = 'Success', int $statusCode = 200): void {
        self::json([
            'success' => true,
            'message' => $message,
            'data' => $data
        ], $statusCode);
    }
    
    /**
     * Error response
     */
    public static function error(string $message, $errors = null, int $statusCode = 400): void {
        $response = [
            'success' => false,
            'message' => $message
        ];
        
        if ($errors !== null) {
            $response['errors'] = $errors;
        }
        
        self::json($response, $statusCode);
    }
    
    /**
     * Validation error response
     */
    public static function validationError(array $errors, string $message = 'Validation failed'): void {
        self::error($message, $errors, 422);
    }
    
    /**
     * Unauthorized response
     */
    public static function unauthorized(string $message = 'Unauthorized'): void {
        self::error($message, null, 401);
    }
    
    /**
     * Forbidden response
     */
    public static function forbidden(string $message = 'Forbidden'): void {
        self::error($message, null, 403);
    }
    
    /**
     * Not found response
     */
    public static function notFound(string $message = 'Resource not found'): void {
        self::error($message, null, 404);
    }
    
    /**
     * Server error response
     */
    public static function serverError(string $message = 'Internal server error'): void {
        self::error($message, null, 500);
    }
    
    /**
     * Paginated response
     */
    public static function paginated(array $data, int $total, int $page, int $perPage): void {
        $totalPages = ceil($total / $perPage);
        
        self::success([
            'items' => $data,
            'pagination' => [
                'total' => $total,
                'per_page' => $perPage,
                'current_page' => $page,
                'total_pages' => $totalPages,
                'has_next' => $page < $totalPages,
                'has_prev' => $page > 1
            ]
        ]);
    }
}
