<?php
/**
 * Validator Class
 * Input validation with comprehensive rules
 */

class Validator {
    private $errors = [];
    private $data;
    
    public function __construct(array $data) {
        $this->data = $data;
    }
    
    /**
     * Validate data against rules
     */
    public function validate(array $rules): bool {
        foreach ($rules as $field => $ruleSet) {
            $fieldRules = is_string($ruleSet) ? explode('|', $ruleSet) : $ruleSet;
            
            foreach ($fieldRules as $rule) {
                $this->applyRule($field, $rule);
            }
        }
        
        return empty($this->errors);
    }
    
    /**
     * Apply single validation rule
     */
    private function applyRule(string $field, string $rule): void {
        $value = $this->data[$field] ?? null;
        
        // Parse rule and parameters
        $params = [];
        if (strpos($rule, ':') !== false) {
            list($rule, $paramStr) = explode(':', $rule, 2);
            $params = explode(',', $paramStr);
        }
        
        $methodName = 'validate' . ucfirst($rule);
        
        if (method_exists($this, $methodName)) {
            $this->$methodName($field, $value, $params);
        }
    }
    
    /**
     * Validate required field
     */
    private function validateRequired(string $field, $value): void {
        if ($value === null || $value === '' || (is_array($value) && empty($value))) {
            $this->addError($field, "{$field} is required");
        }
    }
    
    /**
     * Validate email
     */
    private function validateEmail(string $field, $value): void {
        if ($value && !filter_var($value, FILTER_VALIDATE_EMAIL)) {
            $this->addError($field, "{$field} must be a valid email address");
        }
    }
    
    /**
     * Validate minimum length
     */
    private function validateMin(string $field, $value, array $params): void {
        $min = (int) ($params[0] ?? 0);
        if ($value && strlen($value) < $min) {
            $this->addError($field, "{$field} must be at least {$min} characters");
        }
    }
    
    /**
     * Validate maximum length
     */
    private function validateMax(string $field, $value, array $params): void {
        $max = (int) ($params[0] ?? 0);
        if ($value && strlen($value) > $max) {
            $this->addError($field, "{$field} must not exceed {$max} characters");
        }
    }
    
    /**
     * Validate numeric
     */
    private function validateNumeric(string $field, $value): void {
        if ($value && !is_numeric($value)) {
            $this->addError($field, "{$field} must be numeric");
        }
    }
    
    /**
     * Validate integer
     */
    private function validateInteger(string $field, $value): void {
        if ($value && filter_var($value, FILTER_VALIDATE_INT) === false) {
            $this->addError($field, "{$field} must be an integer");
        }
    }
    
    /**
     * Validate string
     */
    private function validateString(string $field, $value): void {
        if ($value && !is_string($value)) {
            $this->addError($field, "{$field} must be a string");
        }
    }
    
    /**
     * Validate boolean
     */
    private function validateBoolean(string $field, $value): void {
        if ($value !== null && !is_bool($value) && !in_array($value, [0, 1, '0', '1', 'true', 'false'], true)) {
            $this->addError($field, "{$field} must be a boolean");
        }
    }
    
    /**
     * Validate array
     */
    private function validateArray(string $field, $value): void {
        if ($value && !is_array($value)) {
            $this->addError($field, "{$field} must be an array");
        }
    }
    
    /**
     * Validate in list
     */
    private function validateIn(string $field, $value, array $params): void {
        if ($value && !in_array($value, $params)) {
            $list = implode(', ', $params);
            $this->addError($field, "{$field} must be one of: {$list}");
        }
    }
    
    /**
     * Validate regex pattern
     */
    private function validateRegex(string $field, $value, array $params): void {
        $pattern = $params[0] ?? '';
        if ($value && $pattern && !preg_match($pattern, $value)) {
            $this->addError($field, "{$field} format is invalid");
        }
    }
    
    /**
     * Validate phone number (Indonesian format)
     * Supports mobile (08xx, +628xx, 628xx) and landline formats
     */
    private function validatePhone(string $field, $value): void {
        if ($value) {
            $pattern = '/^(\+62|62|0)[0-9]{8,13}$/';
            if (!preg_match($pattern, $value)) {
                $this->addError($field, "{$field} must be a valid Indonesian phone number");
            }
        }
    }
    
    /**
     * Validate date format
     */
    private function validateDate(string $field, $value, array $params): void {
        $format = $params[0] ?? 'Y-m-d';
        if ($value) {
            $d = DateTime::createFromFormat($format, $value);
            if (!$d || $d->format($format) !== $value) {
                $this->addError($field, "{$field} must be a valid date in format {$format}");
            }
        }
    }
    
    /**
     * Validate URL
     */
    private function validateUrl(string $field, $value): void {
        if ($value && !filter_var($value, FILTER_VALIDATE_URL)) {
            $this->addError($field, "{$field} must be a valid URL");
        }
    }
    
    /**
     * Validate UUID
     */
    private function validateUuid(string $field, $value): void {
        if ($value) {
            $pattern = '/^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i';
            if (!preg_match($pattern, $value)) {
                $this->addError($field, "{$field} must be a valid UUID");
            }
        }
    }
    
    /**
     * Validate latitude
     */
    private function validateLatitude(string $field, $value): void {
        if ($value && !is_numeric($value)) {
            $this->addError($field, "{$field} must be a valid latitude");
        } elseif ($value && ($value < -90 || $value > 90)) {
            $this->addError($field, "{$field} must be between -90 and 90");
        }
    }
    
    /**
     * Validate longitude
     */
    private function validateLongitude(string $field, $value): void {
        if ($value && !is_numeric($value)) {
            $this->addError($field, "{$field} must be a valid longitude");
        } elseif ($value && ($value < -180 || $value > 180)) {
            $this->addError($field, "{$field} must be between -180 and 180");
        }
    }
    
    /**
     * Add validation error
     */
    private function addError(string $field, string $message): void {
        if (!isset($this->errors[$field])) {
            $this->errors[$field] = [];
        }
        $this->errors[$field][] = $message;
    }
    
    /**
     * Get validation errors
     */
    public function errors(): array {
        return $this->errors;
    }
    
    /**
     * Check if validation failed
     */
    public function fails(): bool {
        return !empty($this->errors);
    }
    
    /**
     * Check if validation passed
     */
    public function passes(): bool {
        return empty($this->errors);
    }
    
    /**
     * Get first error message
     */
    public function firstError(): ?string {
        if (empty($this->errors)) {
            return null;
        }
        
        $firstField = array_key_first($this->errors);
        return $this->errors[$firstField][0] ?? null;
    }
    
    /**
     * Static validate method
     */
    public static function make(array $data, array $rules): self {
        $validator = new self($data);
        $validator->validate($rules);
        return $validator;
    }
}
