<?php
/**
 * Customer Repository
 * Handles database operations for customers
 */
class CustomerRepository
{
    private $db;
    private $table = 'customers';

    public function __construct()
    {
        $this->db = Database::getInstance();
    }

    /**
     * Find customer by ID
     */
    public function findById($id)
    {
        $query = "SELECT * FROM {$this->table} WHERE id = ? AND deleted_at IS NULL";
        $result = $this->db->query($query, [$id]);
        
        if (empty($result)) {
            return null;
        }
        
        return new Customer($result[0]);
    }

    /**
     * Find customer by user ID
     */
    public function findByUserId($userId)
    {
        $query = "SELECT * FROM {$this->table} WHERE user_id = ? AND deleted_at IS NULL";
        $result = $this->db->query($query, [$userId]);
        
        if (empty($result)) {
            return null;
        }
        
        return new Customer($result[0]);
    }

    /**
     * Find customer by customer code
     */
    public function findByCustomerCode($code)
    {
        $query = "SELECT * FROM {$this->table} WHERE customer_code = ? AND deleted_at IS NULL";
        $result = $this->db->query($query, [$code]);
        
        if (empty($result)) {
            return null;
        }
        
        return new Customer($result[0]);
    }

    /**
     * Create new customer
     */
    public function create($data)
    {
        $id = $this->db->generateUUID();
        $now = date('Y-m-d H:i:s');
        
        // Generate customer code
        $customerCode = $this->generateCustomerCode();
        
        $query = "INSERT INTO {$this->table} 
                  (id, user_id, customer_code, full_name, phone, address, city, province, 
                   postal_code, latitude, longitude, status, created_at, updated_at)
                  VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $params = [
            $id,
            $data['user_id'],
            $customerCode,
            $data['full_name'],
            $data['phone'] ?? null,
            $data['address'] ?? null,
            $data['city'] ?? null,
            $data['province'] ?? null,
            $data['postal_code'] ?? null,
            $data['latitude'] ?? null,
            $data['longitude'] ?? null,
            $data['status'] ?? 'active',
            $now,
            $now
        ];
        
        $this->db->execute($query, $params);
        
        return $this->findById($id);
    }

    /**
     * Update customer
     */
    public function update($id, $data)
    {
        $now = date('Y-m-d H:i:s');
        $fields = [];
        $params = [];
        
        $allowedFields = ['full_name', 'phone', 'address', 'city', 'province', 
                          'postal_code', 'latitude', 'longitude', 'status'];
        
        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $fields[] = "$field = ?";
                $params[] = $data[$field];
            }
        }
        
        if (empty($fields)) {
            return $this->findById($id);
        }
        
        $fields[] = "updated_at = ?";
        $params[] = $now;
        $params[] = $id;
        
        $query = "UPDATE {$this->table} SET " . implode(', ', $fields) . " WHERE id = ?";
        $this->db->execute($query, $params);
        
        return $this->findById($id);
    }

    /**
     * Delete customer (soft delete)
     */
    public function delete($id)
    {
        $now = date('Y-m-d H:i:s');
        $query = "UPDATE {$this->table} SET deleted_at = ?, updated_at = ? WHERE id = ?";
        return $this->db->execute($query, [$now, $now, $id]);
    }

    /**
     * Get all customers with pagination
     */
    public function getAll($page = 1, $perPage = 20, $filters = [])
    {
        $offset = ($page - 1) * $perPage;
        $where = ["deleted_at IS NULL"];
        $params = [];
        
        if (!empty($filters['status'])) {
            $where[] = "status = ?";
            $params[] = $filters['status'];
        }
        
        if (!empty($filters['city'])) {
            $where[] = "city = ?";
            $params[] = $filters['city'];
        }
        
        if (!empty($filters['search'])) {
            $where[] = "(full_name LIKE ? OR customer_code LIKE ? OR phone LIKE ?)";
            $searchTerm = "%{$filters['search']}%";
            $params[] = $searchTerm;
            $params[] = $searchTerm;
            $params[] = $searchTerm;
        }
        
        $whereClause = implode(' AND ', $where);
        
        $query = "SELECT * FROM {$this->table} WHERE {$whereClause} 
                  ORDER BY created_at DESC LIMIT ? OFFSET ?";
        $params[] = $perPage;
        $params[] = $offset;
        
        $results = $this->db->query($query, $params);
        $customers = [];
        
        foreach ($results as $row) {
            $customers[] = new Customer($row);
        }
        
        return $customers;
    }

    /**
     * Count customers
     */
    public function count($filters = [])
    {
        $where = ["deleted_at IS NULL"];
        $params = [];
        
        if (!empty($filters['status'])) {
            $where[] = "status = ?";
            $params[] = $filters['status'];
        }
        
        if (!empty($filters['city'])) {
            $where[] = "city = ?";
            $params[] = $filters['city'];
        }
        
        if (!empty($filters['search'])) {
            $where[] = "(full_name LIKE ? OR customer_code LIKE ? OR phone LIKE ?)";
            $searchTerm = "%{$filters['search']}%";
            $params[] = $searchTerm;
            $params[] = $searchTerm;
            $params[] = $searchTerm;
        }
        
        $whereClause = implode(' AND ', $where);
        $query = "SELECT COUNT(*) as total FROM {$this->table} WHERE {$whereClause}";
        $result = $this->db->query($query, $params);
        
        return $result[0]['total'] ?? 0;
    }

    /**
     * Generate unique customer code
     */
    private function generateCustomerCode()
    {
        $prefix = 'CUST';
        $year = date('Y');
        $month = date('m');
        
        // Get the last customer code for this month
        $query = "SELECT customer_code FROM {$this->table} 
                  WHERE customer_code LIKE ? 
                  ORDER BY customer_code DESC LIMIT 1";
        $pattern = "{$prefix}{$year}{$month}%";
        $result = $this->db->query($query, [$pattern]);
        
        if (empty($result)) {
            $sequence = 1;
        } else {
            $lastCode = $result[0]['customer_code'];
            $sequence = (int)substr($lastCode, -4) + 1;
        }
        
        return sprintf("%s%s%s%04d", $prefix, $year, $month, $sequence);
    }

    /**
     * Get customers by location (for collector)
     */
    public function getByLocation($latitude, $longitude, $radiusKm = 10)
    {
        // Using Haversine formula to find customers within radius
        $query = "SELECT *, 
                  (6371 * acos(cos(radians(?)) * cos(radians(latitude)) * 
                  cos(radians(longitude) - radians(?)) + sin(radians(?)) * 
                  sin(radians(latitude)))) AS distance 
                  FROM {$this->table} 
                  WHERE deleted_at IS NULL AND status = 'active'
                  HAVING distance < ? 
                  ORDER BY distance";
        
        $results = $this->db->query($query, [$latitude, $longitude, $latitude, $radiusKm]);
        $customers = [];
        
        foreach ($results as $row) {
            $customers[] = new Customer($row);
        }
        
        return $customers;
    }
}
