<?php
/**
 * User Repository
 * Handles database operations for users
 */
class UserRepository
{
    private $db;
    private $table = 'users';

    public function __construct()
    {
        $this->db = Database::getInstance();
    }

    /**
     * Find user by ID
     */
    public function findById($id)
    {
        $query = "SELECT * FROM {$this->table} WHERE id = ? AND deleted_at IS NULL";
        $result = $this->db->query($query, [$id]);
        
        if (empty($result)) {
            return null;
        }
        
        return new User($result[0]);
    }

    /**
     * Find user by email
     */
    public function findByEmail($email)
    {
        $query = "SELECT * FROM {$this->table} WHERE email = ? AND deleted_at IS NULL";
        $result = $this->db->query($query, [$email]);
        
        if (empty($result)) {
            return null;
        }
        
        return new User($result[0]);
    }

    /**
     * Find user with role and permissions
     */
    public function findByIdWithPermissions($id)
    {
        $user = $this->findById($id);
        
        if (!$user) {
            return null;
        }

        // Get role
        $roleQuery = "SELECT * FROM roles WHERE id = ?";
        $roleResult = $this->db->query($roleQuery, [$user->getRoleId()]);
        
        if (!empty($roleResult)) {
            $user->setRole($roleResult[0]);
        }

        // Get permissions
        $permQuery = "SELECT p.* FROM permissions p
                      INNER JOIN role_permissions rp ON p.id = rp.permission_id
                      WHERE rp.role_id = ?";
        $permissions = $this->db->query($permQuery, [$user->getRoleId()]);
        
        if (!empty($permissions)) {
            $user->setPermissions(array_column($permissions, 'name'));
        }

        return $user;
    }

    /**
     * Create new user
     */
    public function create($data)
    {
        $id = $this->db->generateUUID();
        $now = date('Y-m-d H:i:s');
        
        $query = "INSERT INTO {$this->table} 
                  (id, email, password, full_name, phone, role_id, status, created_at, updated_at)
                  VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $params = [
            $id,
            $data['email'],
            $data['password'],
            $data['full_name'],
            $data['phone'] ?? null,
            $data['role_id'],
            $data['status'] ?? 'active',
            $now,
            $now
        ];
        
        $this->db->execute($query, $params);
        
        return $this->findById($id);
    }

    /**
     * Update user
     */
    public function update($id, $data)
    {
        $now = date('Y-m-d H:i:s');
        $fields = [];
        $params = [];
        
        $allowedFields = ['email', 'password', 'full_name', 'phone', 'role_id', 'status'];
        
        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $fields[] = "$field = ?";
                $params[] = $data[$field];
            }
        }
        
        if (empty($fields)) {
            return $this->findById($id);
        }
        
        $fields[] = "updated_at = ?";
        $params[] = $now;
        $params[] = $id;
        
        $query = "UPDATE {$this->table} SET " . implode(', ', $fields) . " WHERE id = ?";
        $this->db->execute($query, $params);
        
        return $this->findById($id);
    }

    /**
     * Delete user (soft delete)
     */
    public function delete($id)
    {
        $now = date('Y-m-d H:i:s');
        $query = "UPDATE {$this->table} SET deleted_at = ?, updated_at = ? WHERE id = ?";
        return $this->db->execute($query, [$now, $now, $id]);
    }

    /**
     * Get all users with pagination
     */
    public function getAll($page = 1, $perPage = 20, $filters = [])
    {
        $offset = ($page - 1) * $perPage;
        $where = ["deleted_at IS NULL"];
        $params = [];
        
        if (!empty($filters['status'])) {
            $where[] = "status = ?";
            $params[] = $filters['status'];
        }
        
        if (!empty($filters['role_id'])) {
            $where[] = "role_id = ?";
            $params[] = $filters['role_id'];
        }
        
        $whereClause = implode(' AND ', $where);
        
        $query = "SELECT * FROM {$this->table} WHERE {$whereClause} 
                  ORDER BY created_at DESC LIMIT ? OFFSET ?";
        $params[] = $perPage;
        $params[] = $offset;
        
        $results = $this->db->query($query, $params);
        $users = [];
        
        foreach ($results as $row) {
            $users[] = new User($row);
        }
        
        return $users;
    }

    /**
     * Count users
     */
    public function count($filters = [])
    {
        $where = ["deleted_at IS NULL"];
        $params = [];
        
        if (!empty($filters['status'])) {
            $where[] = "status = ?";
            $params[] = $filters['status'];
        }
        
        if (!empty($filters['role_id'])) {
            $where[] = "role_id = ?";
            $params[] = $filters['role_id'];
        }
        
        $whereClause = implode(' AND ', $where);
        $query = "SELECT COUNT(*) as total FROM {$this->table} WHERE {$whereClause}";
        $result = $this->db->query($query, $params);
        
        return $result[0]['total'] ?? 0;
    }

    /**
     * Record login attempt
     */
    public function recordLoginAttempt($userId, $ipAddress, $userAgent, $success)
    {
        $id = $this->db->generateUUID();
        $query = "INSERT INTO login_attempts 
                  (id, user_id, ip_address, user_agent, success, created_at)
                  VALUES (?, ?, ?, ?, ?, ?)";
        
        $this->db->execute($query, [
            $id,
            $userId,
            $ipAddress,
            $userAgent,
            $success ? 1 : 0,
            date('Y-m-d H:i:s')
        ]);
    }

    /**
     * Get failed login attempts count
     */
    public function getFailedLoginAttempts($userId, $minutes = 15)
    {
        $since = date('Y-m-d H:i:s', strtotime("-{$minutes} minutes"));
        $query = "SELECT COUNT(*) as count FROM login_attempts 
                  WHERE user_id = ? AND success = 0 AND created_at > ?";
        $result = $this->db->query($query, [$userId, $since]);
        
        return $result[0]['count'] ?? 0;
    }

    /**
     * Clear login attempts
     */
    public function clearLoginAttempts($userId)
    {
        $query = "DELETE FROM login_attempts WHERE user_id = ?";
        return $this->db->execute($query, [$userId]);
    }
}
