<?php
/**
 * Billing Service
 * Handles billing and invoice generation logic
 */
class BillingService
{
    private $invoiceRepo;
    private $customerRepo;
    private $logger;

    public function __construct()
    {
        $this->invoiceRepo = new InvoiceRepository();
        $this->customerRepo = new CustomerRepository();
        $this->logger = Logger::getInstance();
    }

    /**
     * Generate invoice for customer
     */
    public function generateInvoice($data)
    {
        try {
            // Validate customer exists
            $customer = $this->customerRepo->findById($data['customer_id']);
            
            if (!$customer) {
                throw new NotFoundException('Customer not found');
            }

            // Calculate amounts
            $subtotal = $data['subtotal'];
            $discountAmount = $data['discount_amount'] ?? 0;
            $taxAmount = $data['tax_amount'] ?? 0;
            $totalAmount = $subtotal - $discountAmount + $taxAmount;

            $invoiceData = [
                'customer_id' => $data['customer_id'],
                'billing_cycle_id' => $data['billing_cycle_id'] ?? null,
                'issue_date' => $data['issue_date'] ?? date('Y-m-d'),
                'due_date' => $data['due_date'],
                'subtotal' => $subtotal,
                'discount_amount' => $discountAmount,
                'tax_amount' => $taxAmount,
                'total_amount' => $totalAmount,
                'status' => 'pending',
                'items' => $data['items'] ?? []
            ];

            $invoice = $this->invoiceRepo->create($invoiceData);

            $this->logger->info('Invoice generated', [
                'invoice_id' => $invoice->getId(),
                'customer_id' => $data['customer_id'],
                'total' => $totalAmount
            ]);

            return $invoice;

        } catch (NotFoundException $e) {
            throw $e;
        } catch (Exception $e) {
            $this->logger->error('Invoice generation failed', [
                'customer_id' => $data['customer_id'] ?? 'unknown',
                'error' => $e->getMessage()
            ]);
            throw new DatabaseException('Failed to generate invoice');
        }
    }

    /**
     * Get invoice by ID
     */
    public function getInvoice($invoiceId)
    {
        $invoice = $this->invoiceRepo->findById($invoiceId);
        
        if (!$invoice) {
            throw new NotFoundException('Invoice not found');
        }

        return $invoice;
    }

    /**
     * Get customer invoices
     */
    public function getCustomerInvoices($customerId, $page = 1, $perPage = 20, $filters = [])
    {
        $invoices = $this->invoiceRepo->getByCustomerId($customerId, $page, $perPage, $filters);
        $total = $this->invoiceRepo->count(array_merge($filters, ['customer_id' => $customerId]));

        return [
            'data' => array_map(function($invoice) {
                return $invoice->toArray();
            }, $invoices),
            'pagination' => [
                'current_page' => $page,
                'per_page' => $perPage,
                'total' => $total,
                'total_pages' => ceil($total / $perPage)
            ]
        ];
    }

    /**
     * Mark invoice as paid
     */
    public function markAsPaid($invoiceId)
    {
        try {
            $invoice = $this->invoiceRepo->findById($invoiceId);
            
            if (!$invoice) {
                throw new NotFoundException('Invoice not found');
            }

            if ($invoice->isPaid()) {
                throw new ValidationException('Invoice is already paid');
            }

            $updated = $this->invoiceRepo->markAsPaid($invoiceId);

            $this->logger->info('Invoice marked as paid', ['invoice_id' => $invoiceId]);

            return $updated;

        } catch (NotFoundException | ValidationException $e) {
            throw $e;
        } catch (Exception $e) {
            $this->logger->error('Failed to mark invoice as paid', [
                'invoice_id' => $invoiceId,
                'error' => $e->getMessage()
            ]);
            throw new DatabaseException('Failed to update invoice status');
        }
    }

    /**
     * Process overdue invoices
     */
    public function processOverdueInvoices()
    {
        try {
            $overdueInvoices = $this->invoiceRepo->getOverdueInvoices();
            $count = 0;

            foreach ($overdueInvoices as $invoice) {
                $this->invoiceRepo->markAsOverdue($invoice->getId());
                $count++;
                
                $this->logger->info('Invoice marked as overdue', [
                    'invoice_id' => $invoice->getId(),
                    'due_date' => $invoice->getDueDate()
                ]);
            }

            return $count;

        } catch (Exception $e) {
            $this->logger->error('Failed to process overdue invoices', [
                'error' => $e->getMessage()
            ]);
            throw new DatabaseException('Failed to process overdue invoices');
        }
    }

    /**
     * Get all invoices with pagination
     */
    public function getAllInvoices($page = 1, $perPage = 20, $filters = [])
    {
        $invoices = $this->invoiceRepo->getAll($page, $perPage, $filters);
        $total = $this->invoiceRepo->count($filters);

        return [
            'data' => array_map(function($invoice) {
                return $invoice->toArray();
            }, $invoices),
            'pagination' => [
                'current_page' => $page,
                'per_page' => $perPage,
                'total' => $total,
                'total_pages' => ceil($total / $perPage)
            ]
        ];
    }

    /**
     * Get dashboard statistics
     */
    public function getDashboardStats()
    {
        $db = Database::getInstance();
        
        // Total invoices by status
        $query = "SELECT 
                    COUNT(*) as total,
                    SUM(CASE WHEN status = 'paid' THEN 1 ELSE 0 END) as paid,
                    SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
                    SUM(CASE WHEN status = 'overdue' THEN 1 ELSE 0 END) as overdue,
                    SUM(total_amount) as total_amount,
                    SUM(CASE WHEN status = 'paid' THEN total_amount ELSE 0 END) as paid_amount
                  FROM invoices 
                  WHERE deleted_at IS NULL";
        
        $result = $db->query($query);
        
        return $result[0] ?? [
            'total' => 0,
            'paid' => 0,
            'pending' => 0,
            'overdue' => 0,
            'total_amount' => 0,
            'paid_amount' => 0
        ];
    }
}
