<?php
/**
 * Invoice PDF Service
 * Generates PDF invoices using TCPDF or similar library
 */
class InvoicePdfService
{
    private $logger;

    public function __construct()
    {
        $this->logger = Logger::getInstance();
    }

    /**
     * Generate invoice PDF
     */
    public function generateInvoicePdf($invoice)
    {
        try {
            // Get invoice data
            $customer = $invoice->getCustomer();
            $items = $invoice->getItems();
            
            // Create HTML content
            $html = $this->generateInvoiceHtml($invoice, $customer, $items);
            
            // Check if we have a PDF library available
            // For now, we'll use a simple HTML to PDF approach
            // In production, you should use a library like TCPDF, mPDF, or Dompdf
            
            // Simple PDF generation (basic approach)
            // This is a placeholder - in production use proper PDF library
            $pdfContent = $this->htmlToPdf($html);
            
            return $pdfContent;

        } catch (Exception $e) {
            $this->logger->error('Generate invoice PDF error', [
                'error' => $e->getMessage(),
                'invoice_id' => $invoice->getId()
            ]);
            throw $e;
        }
    }

    /**
     * Generate invoice HTML
     */
    private function generateInvoiceHtml($invoice, $customer, $items)
    {
        $invoiceNumber = $invoice->getInvoiceNumber();
        $invoiceDate = date('d/m/Y', strtotime($invoice->getCreatedAt()));
        $dueDate = date('d/m/Y', strtotime($invoice->getDueDate()));
        $billingPeriod = date('F Y', strtotime($invoice->getBillingPeriodStart()));
        
        $html = '
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <title>Invoice ' . htmlspecialchars($invoiceNumber) . '</title>
            <style>
                body {
                    font-family: Arial, sans-serif;
                    font-size: 12px;
                    margin: 0;
                    padding: 20px;
                }
                .header {
                    text-align: center;
                    margin-bottom: 30px;
                    border-bottom: 2px solid #333;
                    padding-bottom: 20px;
                }
                .company-name {
                    font-size: 24px;
                    font-weight: bold;
                    color: #333;
                }
                .invoice-title {
                    font-size: 20px;
                    font-weight: bold;
                    margin-top: 10px;
                }
                .info-section {
                    margin-bottom: 30px;
                }
                .info-row {
                    display: table;
                    width: 100%;
                    margin-bottom: 20px;
                }
                .info-col {
                    display: table-cell;
                    width: 50%;
                    vertical-align: top;
                }
                .label {
                    font-weight: bold;
                    color: #666;
                }
                table {
                    width: 100%;
                    border-collapse: collapse;
                    margin-bottom: 20px;
                }
                th {
                    background-color: #f5f5f5;
                    padding: 10px;
                    text-align: left;
                    border-bottom: 2px solid #333;
                    font-weight: bold;
                }
                td {
                    padding: 8px 10px;
                    border-bottom: 1px solid #ddd;
                }
                .text-right {
                    text-align: right;
                }
                .total-section {
                    margin-top: 20px;
                    text-align: right;
                }
                .total-row {
                    margin-bottom: 5px;
                }
                .total-label {
                    display: inline-block;
                    width: 150px;
                    font-weight: bold;
                }
                .total-amount {
                    display: inline-block;
                    width: 150px;
                    text-align: right;
                }
                .grand-total {
                    font-size: 18px;
                    font-weight: bold;
                    color: #333;
                    padding-top: 10px;
                    border-top: 2px solid #333;
                }
                .footer {
                    margin-top: 50px;
                    text-align: center;
                    color: #666;
                    font-size: 10px;
                }
                .status-badge {
                    display: inline-block;
                    padding: 5px 15px;
                    border-radius: 5px;
                    font-weight: bold;
                    text-transform: uppercase;
                }
                .status-paid {
                    background-color: #4CAF50;
                    color: white;
                }
                .status-pending {
                    background-color: #FFC107;
                    color: #333;
                }
                .status-overdue {
                    background-color: #F44336;
                    color: white;
                }
            </style>
        </head>
        <body>
            <div class="header">
                <div class="company-name">' . htmlspecialchars(COMPANY_NAME ?? 'Billing System') . '</div>
                <div>' . htmlspecialchars(COMPANY_ADDRESS ?? '') . '</div>
                <div>' . htmlspecialchars(COMPANY_PHONE ?? '') . '</div>
                <div class="invoice-title">INVOICE</div>
            </div>

            <div class="info-section">
                <div class="info-row">
                    <div class="info-col">
                        <div><span class="label">Invoice Number:</span> ' . htmlspecialchars($invoiceNumber) . '</div>
                        <div><span class="label">Invoice Date:</span> ' . $invoiceDate . '</div>
                        <div><span class="label">Due Date:</span> ' . $dueDate . '</div>
                        <div><span class="label">Billing Period:</span> ' . $billingPeriod . '</div>
                        <div style="margin-top: 10px;">
                            <span class="label">Status:</span>
                            <span class="status-badge status-' . strtolower($invoice->getStatus()) . '">
                                ' . strtoupper($invoice->getStatus()) . '
                            </span>
                        </div>
                    </div>
                    <div class="info-col">
                        <div class="label">Bill To:</div>
                        <div style="margin-top: 5px;">
                            <div><strong>' . htmlspecialchars($customer['full_name'] ?? '') . '</strong></div>
                            <div>' . htmlspecialchars($customer['customer_code'] ?? '') . '</div>
                            <div>' . htmlspecialchars($customer['address'] ?? '') . '</div>
                            <div>' . htmlspecialchars($customer['phone'] ?? '') . '</div>
                            <div>' . htmlspecialchars($customer['email'] ?? '') . '</div>
                        </div>
                    </div>
                </div>
            </div>

            <table>
                <thead>
                    <tr>
                        <th>Description</th>
                        <th class="text-right">Quantity</th>
                        <th class="text-right">Unit Price</th>
                        <th class="text-right">Amount</th>
                    </tr>
                </thead>
                <tbody>';
        
        $subtotal = 0;
        foreach ($items as $item) {
            $itemTotal = $item['quantity'] * $item['unit_price'];
            $subtotal += $itemTotal;
            
            $html .= '
                    <tr>
                        <td>' . htmlspecialchars($item['description']) . '</td>
                        <td class="text-right">' . number_format($item['quantity'], 0) . '</td>
                        <td class="text-right">Rp ' . number_format($item['unit_price'], 0, ',', '.') . '</td>
                        <td class="text-right">Rp ' . number_format($itemTotal, 0, ',', '.') . '</td>
                    </tr>';
        }
        
        $html .= '
                </tbody>
            </table>

            <div class="total-section">
                <div class="total-row">
                    <span class="total-label">Subtotal:</span>
                    <span class="total-amount">Rp ' . number_format($invoice->getSubtotal(), 0, ',', '.') . '</span>
                </div>';
        
        if ($invoice->getTax() > 0) {
            $html .= '
                <div class="total-row">
                    <span class="total-label">Tax:</span>
                    <span class="total-amount">Rp ' . number_format($invoice->getTax(), 0, ',', '.') . '</span>
                </div>';
        }
        
        if ($invoice->getDiscount() > 0) {
            $html .= '
                <div class="total-row">
                    <span class="total-label">Discount:</span>
                    <span class="total-amount">- Rp ' . number_format($invoice->getDiscount(), 0, ',', '.') . '</span>
                </div>';
        }
        
        $html .= '
                <div class="total-row grand-total">
                    <span class="total-label">Total Amount:</span>
                    <span class="total-amount">Rp ' . number_format($invoice->getTotalAmount(), 0, ',', '.') . '</span>
                </div>
            </div>';
        
        if ($invoice->isPaid()) {
            $html .= '
            <div style="margin-top: 30px; text-align: center; color: #4CAF50; font-weight: bold;">
                PAID - ' . date('d/m/Y', strtotime($invoice->getPaidAt())) . '
            </div>';
        } else {
            $html .= '
            <div style="margin-top: 30px; padding: 15px; background-color: #f5f5f5; border-left: 4px solid #FF9800;">
                <strong>Payment Instructions:</strong><br>
                Please make payment before the due date to avoid service disruption.<br>
                You can pay via our mobile app or contact our customer service.
            </div>';
        }
        
        $html .= '
            <div class="footer">
                <p>Thank you for your business!</p>
                <p>This is a computer-generated document. No signature is required.</p>
            </div>
        </body>
        </html>';
        
        return $html;
    }

    /**
     * Convert HTML to PDF
     * This is a simplified version - in production use a proper PDF library
     */
    private function htmlToPdf($html)
    {
        // For production, install and use a proper PDF library like:
        // - mPDF: composer require mpdf/mpdf
        // - TCPDF: composer require tecnickcom/tcpdf
        // - Dompdf: composer require dompdf/dompdf
        
        // For now, return the HTML content with PDF headers
        // The browser/client can handle this appropriately
        return $html;
    }
}
