<?php
/**
 * Notification Service
 * Handles push notifications via Firebase Cloud Messaging
 */
class NotificationService
{
    private $fcmServerKey;
    private $fcmUrl;
    private $logger;
    private $db;

    public function __construct()
    {
        $this->fcmServerKey = FCM_SERVER_KEY ?? '';
        $this->fcmUrl = 'https://fcm.googleapis.com/fcm/send';
        $this->logger = Logger::getInstance();
        $this->db = Database::getInstance();
    }

    /**
     * Send payment confirmation notification
     */
    public function sendPaymentConfirmation($customerId, $paymentId, $amount)
    {
        try {
            // Get customer FCM token
            $tokens = $this->getCustomerTokens($customerId);
            
            if (empty($tokens)) {
                $this->logger->info('No FCM tokens found for customer', ['customer_id' => $customerId]);
                return false;
            }

            // Get payment details
            $paymentRepo = new PaymentRepository();
            $payment = $paymentRepo->findById($paymentId);
            
            if (!$payment) {
                return false;
            }

            $title = 'Pembayaran Berhasil';
            $body = sprintf(
                'Pembayaran Anda sebesar Rp %s telah diterima. Terima kasih!',
                number_format($amount, 0, ',', '.')
            );

            $notification = [
                'title' => $title,
                'body' => $body,
                'sound' => 'default',
                'badge' => '1',
                'icon' => 'ic_notification',
                'color' => '#4CAF50'
            ];

            $data = [
                'type' => 'payment_success',
                'payment_id' => $paymentId,
                'amount' => $amount,
                'timestamp' => date('c')
            ];

            $result = $this->sendToTokens($tokens, $notification, $data);
            
            // Save notification to database
            $this->saveNotification(
                $customerId,
                'payment_success',
                $title,
                $body,
                $data
            );

            return $result;

        } catch (Exception $e) {
            $this->logger->error('Send payment confirmation error', [
                'error' => $e->getMessage(),
                'customer_id' => $customerId,
                'payment_id' => $paymentId
            ]);
            return false;
        }
    }

    /**
     * Send invoice reminder notification
     */
    public function sendInvoiceReminder($customerId, $invoiceId, $dueDate, $amount)
    {
        try {
            $tokens = $this->getCustomerTokens($customerId);
            
            if (empty($tokens)) {
                return false;
            }

            $daysUntilDue = max(0, (strtotime($dueDate) - time()) / 86400);
            
            $title = 'Pengingat Tagihan';
            $body = sprintf(
                'Tagihan Anda sebesar Rp %s akan jatuh tempo dalam %d hari.',
                number_format($amount, 0, ',', '.'),
                ceil($daysUntilDue)
            );

            $notification = [
                'title' => $title,
                'body' => $body,
                'sound' => 'default',
                'badge' => '1',
                'icon' => 'ic_notification',
                'color' => '#FF9800'
            ];

            $data = [
                'type' => 'invoice_reminder',
                'invoice_id' => $invoiceId,
                'due_date' => $dueDate,
                'amount' => $amount,
                'timestamp' => date('c')
            ];

            $result = $this->sendToTokens($tokens, $notification, $data);
            
            $this->saveNotification(
                $customerId,
                'invoice_reminder',
                $title,
                $body,
                $data
            );

            return $result;

        } catch (Exception $e) {
            $this->logger->error('Send invoice reminder error', [
                'error' => $e->getMessage(),
                'customer_id' => $customerId,
                'invoice_id' => $invoiceId
            ]);
            return false;
        }
    }

    /**
     * Send overdue notification
     */
    public function sendOverdueNotification($customerId, $invoiceId, $amount, $daysOverdue)
    {
        try {
            $tokens = $this->getCustomerTokens($customerId);
            
            if (empty($tokens)) {
                return false;
            }

            $title = 'Tagihan Jatuh Tempo';
            $body = sprintf(
                'Tagihan Anda sebesar Rp %s telah jatuh tempo %d hari yang lalu. Mohon segera melakukan pembayaran.',
                number_format($amount, 0, ',', '.'),
                $daysOverdue
            );

            $notification = [
                'title' => $title,
                'body' => $body,
                'sound' => 'default',
                'badge' => '1',
                'icon' => 'ic_notification',
                'color' => '#F44336'
            ];

            $data = [
                'type' => 'invoice_overdue',
                'invoice_id' => $invoiceId,
                'amount' => $amount,
                'days_overdue' => $daysOverdue,
                'timestamp' => date('c')
            ];

            $result = $this->sendToTokens($tokens, $notification, $data);
            
            $this->saveNotification(
                $customerId,
                'invoice_overdue',
                $title,
                $body,
                $data
            );

            return $result;

        } catch (Exception $e) {
            $this->logger->error('Send overdue notification error', [
                'error' => $e->getMessage(),
                'customer_id' => $customerId,
                'invoice_id' => $invoiceId
            ]);
            return false;
        }
    }

    /**
     * Send notification to multiple tokens
     */
    private function sendToTokens($tokens, $notification, $data = [])
    {
        try {
            $payload = [
                'registration_ids' => $tokens,
                'notification' => $notification,
                'data' => $data,
                'priority' => 'high',
                'content_available' => true
            ];

            $headers = [
                'Authorization: key=' . $this->fcmServerKey,
                'Content-Type: application/json'
            ];

            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $this->fcmUrl);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));

            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);

            if ($error) {
                throw new Exception("CURL Error: {$error}");
            }

            $result = json_decode($response, true);
            
            $this->logger->info('FCM notification sent', [
                'tokens_count' => count($tokens),
                'success_count' => $result['success'] ?? 0,
                'failure_count' => $result['failure'] ?? 0
            ]);

            return $result['success'] > 0;

        } catch (Exception $e) {
            $this->logger->error('Send FCM notification error', ['error' => $e->getMessage()]);
            return false;
        }
    }

    /**
     * Get customer FCM tokens
     */
    private function getCustomerTokens($customerId)
    {
        try {
            $query = "SELECT fcm_token FROM user_devices 
                     WHERE user_id = (SELECT user_id FROM customers WHERE id = ?) 
                     AND fcm_token IS NOT NULL 
                     AND is_active = 1";
            
            $results = $this->db->query($query, [$customerId]);
            
            return array_column($results, 'fcm_token');

        } catch (Exception $e) {
            $this->logger->error('Get customer tokens error', ['error' => $e->getMessage()]);
            return [];
        }
    }

    /**
     * Save notification to database
     */
    private function saveNotification($customerId, $type, $title, $body, $data = [])
    {
        try {
            $query = "INSERT INTO notifications 
                     (customer_id, type, title, body, data, is_read, created_at) 
                     VALUES (?, ?, ?, ?, ?, 0, NOW())";
            
            $this->db->execute($query, [
                $customerId,
                $type,
                $title,
                $body,
                json_encode($data)
            ]);

        } catch (Exception $e) {
            $this->logger->error('Save notification error', ['error' => $e->getMessage()]);
        }
    }

    /**
     * Register FCM token
     */
    public function registerToken($userId, $token, $deviceInfo = [])
    {
        try {
            // Check if token already exists
            $checkQuery = "SELECT id FROM user_devices WHERE fcm_token = ?";
            $existing = $this->db->query($checkQuery, [$token]);

            if (!empty($existing)) {
                // Update existing
                $updateQuery = "UPDATE user_devices 
                               SET user_id = ?, is_active = 1, updated_at = NOW() 
                               WHERE fcm_token = ?";
                $this->db->execute($updateQuery, [$userId, $token]);
            } else {
                // Insert new
                $insertQuery = "INSERT INTO user_devices 
                               (user_id, fcm_token, device_type, device_model, is_active) 
                               VALUES (?, ?, ?, ?, 1)";
                $this->db->execute($insertQuery, [
                    $userId,
                    $token,
                    $deviceInfo['type'] ?? 'mobile',
                    $deviceInfo['model'] ?? 'unknown'
                ]);
            }

            return true;

        } catch (Exception $e) {
            $this->logger->error('Register FCM token error', ['error' => $e->getMessage()]);
            return false;
        }
    }

    /**
     * Unregister FCM token
     */
    public function unregisterToken($token)
    {
        try {
            $query = "UPDATE user_devices SET is_active = 0 WHERE fcm_token = ?";
            $this->db->execute($query, [$token]);
            return true;

        } catch (Exception $e) {
            $this->logger->error('Unregister FCM token error', ['error' => $e->getMessage()]);
            return false;
        }
    }
}
