<?php
/**
 * Tripay Service
 * Integration with Tripay payment gateway
 */
class TripayService
{
    private $apiKey;
    private $privateKey;
    private $merchantCode;
    private $apiUrl;
    private $isSandbox;
    private $logger;

    public function __construct()
    {
        $this->apiKey = defined('TRIPAY_API_KEY') ? TRIPAY_API_KEY : '';
        $this->privateKey = defined('TRIPAY_PRIVATE_KEY') ? TRIPAY_PRIVATE_KEY : '';
        $this->merchantCode = defined('TRIPAY_MERCHANT_CODE') ? TRIPAY_MERCHANT_CODE : '';
        $this->isSandbox = defined('TRIPAY_SANDBOX') ? TRIPAY_SANDBOX : true;
        
        $this->apiUrl = $this->isSandbox 
            ? 'https://tripay.co.id/api-sandbox' 
            : 'https://tripay.co.id/api';
            
        $this->logger = Logger::getInstance();
        
        // Validate required credentials
        if (empty($this->apiKey) || empty($this->privateKey) || empty($this->merchantCode)) {
            $this->logger->warning('Tripay credentials not configured properly');
        }
    }

    /**
     * Get available payment channels
     */
    public function getPaymentChannels()
    {
        try {
            $endpoint = '/merchant/payment-channel';
            $response = $this->makeRequest('GET', $endpoint);
            
            if ($response['success'] && isset($response['data'])) {
                return array_map(function($channel) {
                    return [
                        'code' => $channel['code'],
                        'name' => $channel['name'],
                        'type' => $this->mapChannelType($channel['group']),
                        'icon_url' => $channel['icon_url'] ?? null,
                        'admin_fee' => [
                            'flat' => $channel['total_fee']['flat'] ?? 0,
                            'percent' => $channel['total_fee']['percent'] ?? 0
                        ],
                        'min_amount' => $channel['minimum_fee'] ?? 0,
                        'max_amount' => $channel['maximum_fee'] ?? null,
                        'is_active' => $channel['active'] ?? true
                    ];
                }, $response['data']);
            }
            
            return [];

        } catch (Exception $e) {
            $this->logger->error('Get Tripay payment channels error', ['error' => $e->getMessage()]);
            return [];
        }
    }

    /**
     * Create payment transaction
     */
    public function createTransaction($data)
    {
        try {
            $endpoint = '/transaction/create';
            
            // Calculate admin fee
            $channel = $this->getChannelInfo($data['payment_method']);
            $adminFee = $this->calculateFee($data['amount'], $channel);
            
            $payload = [
                'method' => $data['payment_method'],
                'merchant_ref' => $data['external_id'],
                'amount' => (int)$data['amount'],
                'customer_name' => $data['customer_name'],
                'customer_email' => $data['customer_email'],
                'customer_phone' => $data['customer_phone'] ?? '',
                'order_items' => $data['items'] ?? [
                    [
                        'name' => $data['description'] ?? 'Payment',
                        'price' => (int)$data['amount'],
                        'quantity' => 1
                    ]
                ],
                'return_url' => $data['success_url'] ?? APP_URL . '/payment/success',
                'expired_time' => (time() + ($data['duration'] ?? 86400)), // 24 hours default
                'signature' => $this->generateSignature($data['external_id'], $data['amount'])
            ];
            
            $response = $this->makeRequest('POST', $endpoint, $payload);
            
            if ($response['success'] && isset($response['data'])) {
                $result = $response['data'];
                return [
                    'success' => true,
                    'transaction_id' => $result['reference'],
                    'merchant_ref' => $result['merchant_ref'],
                    'payment_method' => $result['payment_method'],
                    'payment_name' => $result['payment_name'],
                    'amount' => $result['amount'],
                    'fee' => $result['total_fee'],
                    'amount_received' => $result['amount_received'],
                    'checkout_url' => $result['checkout_url'],
                    'qr_url' => $result['qr_url'] ?? null,
                    'qr_string' => $result['qr_string'] ?? null,
                    'pay_code' => $result['pay_code'] ?? null,
                    'pay_url' => $result['pay_url'] ?? null,
                    'expired_time' => $result['expired_time'],
                    'instructions' => $result['instructions'] ?? []
                ];
            }
            
            return [
                'success' => false,
                'message' => $response['message'] ?? 'Failed to create transaction'
            ];

        } catch (Exception $e) {
            $this->logger->error('Create Tripay transaction error', [
                'error' => $e->getMessage(),
                'data' => $data
            ]);
            
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Get transaction status
     */
    public function getTransactionStatus($reference)
    {
        try {
            $endpoint = "/transaction/detail?reference={$reference}";
            $response = $this->makeRequest('GET', $endpoint);
            
            if ($response['success'] && isset($response['data'])) {
                return [
                    'success' => true,
                    'data' => $response['data']
                ];
            }
            
            return [
                'success' => false,
                'message' => $response['message'] ?? 'Transaction not found'
            ];

        } catch (Exception $e) {
            $this->logger->error('Get Tripay transaction status error', ['error' => $e->getMessage()]);
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Verify webhook callback signature
     */
    public function verifyWebhookSignature($signature, $payload)
    {
        try {
            $jsonPayload = is_string($payload) ? $payload : json_encode($payload);
            $calculatedSignature = hash_hmac('sha256', $jsonPayload, $this->privateKey);
            
            return hash_equals($calculatedSignature, $signature);

        } catch (Exception $e) {
            $this->logger->error('Verify Tripay webhook signature error', ['error' => $e->getMessage()]);
            return false;
        }
    }

    /**
     * Generate signature for transaction
     */
    private function generateSignature($merchantRef, $amount)
    {
        $data = $this->merchantCode . $merchantRef . $amount;
        return hash_hmac('sha256', $data, $this->privateKey);
    }

    /**
     * Make HTTP request to Tripay API
     */
    private function makeRequest($method, $endpoint, $data = [])
    {
        $url = $this->apiUrl . $endpoint;
        
        $headers = [
            'Authorization: Bearer ' . $this->apiKey,
            'Content-Type: application/json'
        ];
        
        $ch = curl_init();
        
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, !$this->isSandbox);
        
        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        
        curl_close($ch);
        
        if ($error) {
            throw new Exception("CURL Error: {$error}");
        }
        
        $result = json_decode($response, true);
        
        if ($httpCode >= 400) {
            throw new Exception($result['message'] ?? "HTTP Error: {$httpCode}");
        }
        
        return $result;
    }

    /**
     * Get channel information
     */
    private function getChannelInfo($channelCode)
    {
        $channels = $this->getPaymentChannels();
        
        foreach ($channels as $channel) {
            if ($channel['code'] === $channelCode) {
                return $channel;
            }
        }
        
        return null;
    }

    /**
     * Calculate admin fee
     */
    private function calculateFee($amount, $channel)
    {
        if (!$channel) {
            return 0;
        }
        
        $flat = $channel['admin_fee']['flat'] ?? 0;
        $percent = $channel['admin_fee']['percent'] ?? 0;
        
        return $flat + ($amount * $percent / 100);
    }

    /**
     * Map channel group to type
     */
    private function mapChannelType($group)
    {
        $typeMap = [
            'Virtual Account' => 'virtual_account',
            'E-Wallet' => 'ewallet',
            'Convenience Store' => 'retail',
            'QRIS' => 'qris',
            'Direct Debit' => 'bank_transfer'
        ];
        
        return $typeMap[$group] ?? 'other';
    }
}
